
#include <maya/MPxFileTranslator.h>
#include <maya/MFnDagNode.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MGlobal.h>

#include <MDt.h>
#include <MDtExt.h>

#include "rwcommon.h"
#include <rpmipkl.h>

#include "querymel.h"
#include "material.h"
#include <rpmatfx.h>

RwMatrix *
getUVTransformMatrix(int shape, int group)
{
    int         mtlID;
    char        *mtlName;
    char        *texName = NULL;
    RwMatrix    *uvTrans = RwMatrixCreate();    

    DtMtlGetID(shape, group, &mtlID);    
    DtMtlGetNameByID(mtlID, &mtlName);

    RwMatrixSetIdentity(uvTrans);

    texName = getQueryAsString("listConnections", mtlName, "color", true);
    if (texName)
    {
        RwV3d cov = {1.0f, 1.0f, 1.0f};
        RwV3d trans = {0.0f, 0.0f, 0.0f};        
        float rotFrame = 0.0f;
        RwV3d rep = {1.0f, 1.0f, 1.0f};
        RwV3d off = {0.0f, 0.0f, 0.0f};
        float rotUV = 0.0f;
        RwV3d rotAxis = {0.0f, 0.0f, 1.0f};
        RwV3d centeringOff = {-0.5f, -0.5f, 0.0f};

        /* get values from the texture */
        getQueryAsFloat("getAttr", texName, "coverageU", &cov.x, true);
        getQueryAsFloat("getAttr", texName, "coverageV", &cov.y, true);
        getQueryAsFloat("getAttr", texName, "translateFrameU", &trans.x, true);
        getQueryAsFloat("getAttr", texName, "translateFrameV", &trans.y, true);
        getQueryAsFloat("getAttr", texName, "rotateFrame", &rotFrame, true);
        getQueryAsFloat("getAttr", texName, "repeatU", &rep.x, true);
        getQueryAsFloat("getAttr", texName, "repeatV", &rep.y, true);
        getQueryAsFloat("getAttr", texName, "offsetU", &off.x, true);
        getQueryAsFloat("getAttr", texName, "offsetV", &off.y, true);
        getQueryAsFloat("getAttr", texName, "rotateUV", &rotUV, true);
        
        cov.x = 1.0f / cov.x;
        cov.y = 1.0f / cov.y;
        trans.x = -trans.x;
        trans.y = -trans.y;
        
        RwMatrixTranslate(uvTrans, &centeringOff, rwCOMBINEPOSTCONCAT);
        RwMatrixRotate(uvTrans, &rotAxis, rotFrame, rwCOMBINEPOSTCONCAT);        
        RwV3dScale(&centeringOff, &centeringOff, -1.0f);
        RwMatrixTranslate(uvTrans, &centeringOff, rwCOMBINEPOSTCONCAT);        
        RwMatrixTranslate(uvTrans, &trans, rwCOMBINEPOSTCONCAT);        
        RwMatrixScale(uvTrans, &cov, rwCOMBINEPOSTCONCAT);   

        RwV3dScale(&centeringOff, &centeringOff, -1.0f);
        RwMatrixScale(uvTrans, &rep, rwCOMBINEPOSTCONCAT);            
        RwMatrixTranslate(uvTrans, &off, rwCOMBINEPOSTCONCAT);
        RwMatrixTranslate(uvTrans, &centeringOff, rwCOMBINEPOSTCONCAT);
        RwMatrixRotate(uvTrans, &rotAxis, rotUV, rwCOMBINEPOSTCONCAT); 
        RwV3dScale(&centeringOff, &centeringOff, -1.0f);
        RwMatrixTranslate(uvTrans, &centeringOff, rwCOMBINEPOSTCONCAT);   

        free(texName);
    }
    
    return uvTrans;
}

void
getTexColorGain(char *texFileNodename, float *r, float *g, float *b)
{
    if (texFileNodename)
    {
        getQueryAsFloat("getAttr", texFileNodename, "colorGainR", r, true);
        getQueryAsFloat("getAttr", texFileNodename, "colorGainG", g, true);
        getQueryAsFloat("getAttr", texFileNodename, "colorGainB", b, true);
    }
}

bool
getTexMipmapK(char *texFileNodename, float *k)
{
    bool retVal = false;

    if (texFileNodename)
    {
        float newK;
        
        if (getQueryAsFloat("getAttr", texFileNodename, "RwK", &newK, true))
        {
            *k = newK;
            retVal = true;
        }
    }

    return retVal;
}

bool
getTexMipmapL(char *texFileNodename, RwUInt32 *l)
{
    bool retVal = false;

    if (texFileNodename)
    {
        float newL;
        
        if (getQueryAsFloat("getAttr", texFileNodename, "RwL", &newL, true))
        {
            *l = (RwUInt32)newL;
            retVal = true;
        }
    }

    return retVal;
}

void getTexAddressModes(char *texFileNodename, RwTextureAddressMode *uMode,
                        RwTextureAddressMode *vMode)
{
    int wrapU, wrapV, mirror;

    if (texFileNodename)
    {
        getQueryAsInt("getAttr", texFileNodename, "wrapU", &wrapU, true);
        getQueryAsInt("getAttr", texFileNodename, "wrapV", &wrapV, true);
        getQueryAsInt("getAttr", texFileNodename, "mirror", &mirror, true);
    }

    if (wrapU)
    {
        if (mirror)
        {
            *uMode = rwTEXTUREADDRESSMIRROR;
        }
        else
        {
            *uMode = rwTEXTUREADDRESSWRAP;
        }
    }
    else
    {
        *uMode = rwTEXTUREADDRESSCLAMP;
    }

    if (wrapV)
    {
        if (mirror)
        {
            *vMode = rwTEXTUREADDRESSMIRROR;
        }
        else
        {
            *vMode = rwTEXTUREADDRESSWRAP;
        }
    }
    else
    {
        *vMode = rwTEXTUREADDRESSCLAMP;
    }
}

char *getTexFileNodeName(char *mtlName, char *texType)
//return the name of the file node that represents this texture.
//i.e. NOT the filename of the file texture.
{
    char *filenodeName = getQueryAsString("listConnections", mtlName, texType, true);
    
    if (!filenodeName)
    {
        return 0;    //didn't even have this attribute
    }

    //if it was a bump map we have to go further to find the file
    if (MString(texType) == MString("normalCamera"))
    {
        char *bumpFile = getQueryAsString("listConnections", filenodeName, "bumpValue", true);

        if (bumpFile)
        {
            //and here's the file!
            free( filenodeName );
            filenodeName = bumpFile;
        }
    }
    //if it was an environment map we have to go further to find the file
    else if (MString(texType) == MString("reflectedColor"))
    {
        /*
        //TO DO:
        //could check if the envmap is connected to a bump map like this 
        //(as bumping can affect env mapping, and should if they're used in conjunction,
        //otherwise it looks a bit crappy. Now whether Renderware can manage that!? )
        char *bumpFile = getQueryAsString("listConnections", filenodeName, "normalCamera", true);
        */

        //get the image (texture file) the env map is connected to:
        char *envFile = getQueryAsString("listConnections", filenodeName, "image", true);

        if (envFile)
        {
            //and here's the file!
            free( filenodeName );
            filenodeName = envFile;
        }
    }

    return filenodeName;
}

char *
getTexFilename(char *filenodeName)
//returns the filename of the file used by the file node with the given name.
{   
    char *texPath;
 
    texPath = getQueryAsString("getAttr", filenodeName, "fileTextureName", true);

    if (texPath)
    {
        char *texName;
        char *dotpos = strrchr(texPath, '.');
        
        if (dotpos)
        {
            *dotpos = 0;
        }
        if (strrchr(texPath, '/'))
        
        {
            texName = strdup(strrchr(texPath,'/')+1);
        }
        else
        {
            texName = strdup(texPath);
        }
        
        if (dotpos)
        {
            *dotpos = '.';
        }
        
        free(texPath);

        return texName;
    }
    else
    {
        return NULL;
    }
}

RwTextureFilterMode
getTextureFilterMode(char *texFileNodename)
{
    if (texFileNodename)
    {
        int filterType = 0;

        getQueryAsInt("getAttr", texFileNodename, "filterType", &filterType, true);

        switch (filterType)
        {
            case 0:
                return rwFILTERNEAREST;
            case 1:
                return rwFILTERLINEARMIPLINEAR;
            default:
                return rwFILTERLINEAR;
        }
    }
    else
    {
        return rwFILTERNEAREST;
    }
}

/* Compare two Maya shading groups for equivalence */
bool CompareMayaShadingGroups(int id0, int id1)
{
    char    *mtlName0, *mtlName1;
    char    *texName0, *texName1;
    char    *maskName0, *maskName1;
    float   r0, g0, b0, t0 = 0.0f;
    float   r1, g1, b1, t1 = 0.0f;
    bool   returnVal = true;
    
    DtMtlGetNameByID(id0, &mtlName0);
    DtMtlGetNameByID(id1, &mtlName1);

    texName0    = getTexFileNodeName(mtlName0, "color");
    maskName0   = getTexFileNodeName(mtlName0, "transparency");
 
    texName1    = getTexFileNodeName(mtlName1, "color");
    maskName1   = getTexFileNodeName(mtlName1, "transparency");

    if (texName0 == NULL)
    {
        DtMtlGetDiffuseClr(mtlName0, 0, &r0, &g0, &b0);
    }
    else
    {
        getTexColorGain(texName0, &r0, &g0, &b0);
    }
    
    if (maskName0 == NULL)
    {
        DtMtlGetTransparency(mtlName0, 0, &t0);
    }

    if (texName1 == NULL)
    {
        DtMtlGetDiffuseClr(mtlName1, 0, &r1, &g1, &b1);
    }
    else
    {
        getTexColorGain(texName1, &r1, &g1, &b1);
    }

    if (maskName1 == NULL)
    {
        DtMtlGetTransparency(mtlName1, 0, &t1);
    }

    /* If any colors are different then fail */
    if ((r0 != r1) || (g0 != g1) || (b0 != b1))
    {
        returnVal = false;
        goto end;
    }

    /* If one shading group has a texture and the other doesn't then fail */
    if ((texName0 && !texName1) || (!texName0 && texName1))
    {
        returnVal = false;
        goto end;
    }

    if (texName0)
    {
        /*
            Check if the texture names are the same. Note that we don't check
            the mask names, filter modes or addressing modes as RenderWare
            texture dictionaries only differentiate on the texture name.
        */
        if (stricmp(texName0, texName1) != 0)
        {
            returnVal = false;
            goto end;
        }
    }
    else
    {
        /* Check if the transparency color is the same */
        if (t0 != t1)
        {
            returnVal = false;
            goto end;
        }
    }

end:
    if (texName0)
    {
        RwFree(texName0);
    }

    if (texName1)
    {
        RwFree(texName1);
    }

    if (maskName0)
    {
        RwFree(maskName0);
    }

    if (maskName1)
    {
        RwFree(maskName1);
    }

    return returnVal;
}

RwTexture *
MayaTexture2RwTexture( char *mtlName, char *texType, RwRGBA &color, bool *textureNameWarning )
{
    char *texFilenodeName = getTexFileNodeName(mtlName, texType);
    if (!texFilenodeName)
    {
        //didn't have this kind of texture
        return 0;
    }

    char *texName = getTexFilename(texFilenodeName);
    if (!texName)
    {
        //not a filenode after all!
        //might be a layered texture used as a color for example.
        //(or somebody made a shader network we weren't expecting.)
        return 0;
    }

    //issue a warning if texture name was too long
    if (strlen(texName) >= rwTEXTUREBASENAMELENGTH)
    {
        printf("Warning: Texture name %s has been truncated to ", texName);

        texName[rwTEXTUREBASENAMELENGTH - 1] = 0;
 
        printf("%s\n", texName);

        if (*textureNameWarning)
        {
            MessageBox(NULL, "Warning: Texture names have been truncated.\nPlease see the Maya output window for more details", "RenderWare Exporter", MB_OK);
            *textureNameWarning = false;
        }
    }

    //Texture's color - most of the time, this will be ignored.
    //Maybe issue a warning if it's not set to the defaults?
    float r = 1.0f, g = 1.0f, b = 1.0f, t = 0.0f;
    getTexColorGain(texFilenodeName, &r, &g, &b);
 
    //Texture's alpha - often may be ignored by Renderware, bump mapping may use a diffuse texture's alpha as bump info.
    //Maybe issue some warnings if texture isn't a diffuse or a bump?
    char *maskFilenodeName = getTexFileNodeName(mtlName, "transparency");
    char *maskName = 0;

    if (maskFilenodeName)
    {
        maskName = getTexFilename( maskFilenodeName );
    }

    if (maskName && (strlen(maskName) >= rwTEXTUREBASENAMELENGTH))
    {
        printf("Warning: Texture mask name %s has been truncated to ", maskName);

        maskName[rwTEXTUREBASENAMELENGTH - 1] = 0;
        
        printf("%s\n", maskName);

        if (*textureNameWarning)
        {
            MessageBox(NULL, "Warning: Texture names have been truncated.\nPlease see the Maya output window for more details", "RenderWare Exporter", MB_OK);
            *textureNameWarning = false;
        }
    }

    //Just a constant alpha
    if (!maskName)
    {            
        DtMtlGetTransparency(mtlName, 0, &t);            
    }

    //now we know the color set by the user - convert to RwColor. 
    color.red   = (RwUInt8) (r * 255.0f); 
    color.green = (RwUInt8) (g * 255.0f); 
    color.blue  = (RwUInt8) (b * 255.0f); 
    color.alpha = 255 - (RwUInt8) (t * 255.0f);

    //fetch filtering modes
    RwTextureFilterMode filterMode;
    RwInt32 flags = 0;

    filterMode = getTextureFilterMode(texFilenodeName);
    if (maskName)
    {
        flags |= rwRASTERFORMAT8888;
    }
    else
    {
        flags |= rwRASTERFORMAT888;
    }
    if (filterMode == rwFILTERMIPNEAREST ||
        filterMode == rwFILTERMIPLINEAR ||
        filterMode == rwFILTERLINEARMIPNEAREST ||
        filterMode == rwFILTERLINEARMIPLINEAR)
    {
        flags |= rwRASTERFORMATMIPMAP | rwRASTERFORMATAUTOMIPMAP;
    }
    RwRaster *raster = RwRasterCreate(0, 0, 0, flags);

    RwTexture *texture = RwTextureCreate(raster);
    if (texture)
    {
        RwTextureSetName(texture, texName);
        free(texName);
        if (maskName)
        {
            RwTextureSetMaskName(texture, maskName);
            free(maskName);
        }
     

        RwTextureSetFilterMode(texture, filterMode);

        //fetch & set addressing modes
        RwTextureAddressMode uMode, vMode;
        getTexAddressModes(texFilenodeName, &uMode, &vMode);
        RwTextureSetAddressingU(texture, uMode);
        RwTextureSetAddressingV(texture, vMode);

        //fetch & set K & L values
        float       k;
        RwUInt32    l;

        if (getTexMipmapK(texFilenodeName, &k))
        {
            RpMipmapKLTextureSetK(texture, k);
        }
        
        if (getTexMipmapL(texFilenodeName, &l))
        {
            RpMipmapKLTextureSetL(texture, l);
        }
    }

    return texture;
}

RpMaterial *
MayaShadingGroup2RpMaterial(int materialID, bool verbose, bool *textureNameWarning)
{  
    char        *mtlName;
    DtMtlGetNameByID(materialID, &mtlName);

    RpMaterial  *mat = RpMaterialCreate();

    RwRGBA      color;

    //get the diffuse texture & set material's color from that.
    RwTexture   *diffuseTexture = MayaTexture2RwTexture( mtlName, "color", color, textureNameWarning );

    if (diffuseTexture)
    {
        RpMaterialSetColor(mat, &color);
        RpMaterialSetTexture(mat, diffuseTexture);
        RwTextureDestroy(diffuseTexture);
    }
    else
    {
        //no diffuse texture, just get the material's color
        float   r, g, b, t = 0.0f;
        DtMtlGetDiffuseClr(mtlName, 0, &r, &g, &b);
        DtMtlGetTransparency(mtlName, 0, &t);
        color.red   = (RwUInt8) (r * 255.0f); 
        color.green = (RwUInt8) (g * 255.0f); 
        color.blue  = (RwUInt8) (b * 255.0f); 
        color.alpha = 255 - (RwUInt8) (t * 255.0f);
        RpMaterialSetColor(mat, &color);
    }

    //is there a bump texture?
    RwTexture *bumpTexture = MayaTexture2RwTexture( mtlName, "normalCamera", color, textureNameWarning );

    //is there an environment texture?
    RwTexture *envTexture = MayaTexture2RwTexture( mtlName, "reflectedColor", color, textureNameWarning );
    
    /* 
        Set the material effects we need. RpMatFXMaterialSetEffects actually allocates the
        material effect data and therefore resets any state - hence we need to do it first.
    */
    if (bumpTexture && envTexture)
    {
        RpMatFXMaterialSetEffects(mat, rpMATFXEFFECTBUMPENVMAP);
    }
    else if(bumpTexture)
    {
        RpMatFXMaterialSetEffects(mat, rpMATFXEFFECTBUMPMAP);
    }
    else if(envTexture)
    {
        RpMatFXMaterialSetEffects(mat, rpMATFXEFFECTENVMAP);
    }

    if (bumpTexture)
    {
        RpMatFXMaterialSetBumpMapTexture(mat, bumpTexture);

        RwTextureDestroy(bumpTexture);

        //what's the bump texture's height?
        //get the bump node again
        char *bumpNodeName = getQueryAsString("listConnections", mtlName, "normalCamera", true);

        char *bumpDepthFileNode = getQueryAsString("listConnections", bumpNodeName, "bumpDepth", true);

        if (bumpDepthFileNode)
        {
            //this is the name of a texture file node (or something) that controls the bump depth!

        }
        else
        {
            //bump depth is just a constant
            float bumpDepth;
            
            if (getQueryAsFloat("getAttr", bumpNodeName, "bumpDepth", &bumpDepth, true))
            {
                RpMatFXMaterialSetBumpMapCoefficient(mat, bumpDepth);
            }
        }
    }

    if (envTexture)
    {
        RpMatFXMaterialSetEnvMapTexture( mat, envTexture );

        RwTextureDestroy(envTexture);

        //what's the env texture's reflectivity?
        //get the env node again
        char *reflectivityFileNode = getQueryAsString("listConnections", mtlName, "reflectivity", true);

        if (reflectivityFileNode)
        {
            //this is the name of a texture file node (or something) that controls the reflectivity!

        }
        else
        {
            //reflectivity is just a constant between 0 and 1.
            float reflectivity;
            
            if (getQueryAsFloat("getAttr", mtlName, "reflectivity", &reflectivity, true))
            {
                RpMatFXMaterialSetEnvMapCoefficient(mat, reflectivity);
            }
        }
    }

    //are there any layered textures for second pass?

    return mat;
}

bool
MayaShadingGroupIsTextured(int shapeIndex, int group)
{
    int     mtlID;
    char    *mtlName;
    char    *texName = NULL;
    
    DtMtlGetID(shapeIndex, group, &mtlID);    
    DtMtlGetNameByID(mtlID, &mtlName);

    texName = getTexFileNodeName(mtlName, "color");
    if (texName)
    {
        RwFree(texName);
        return true;
    }
    else
    {
        return false;
    }
}

void
CheckTextureMask(RpMaterial *material, int shape, int group)
{
    int     mtlID;
    char    *mtlName;
    char    *texName = NULL;
    char    *maskName = NULL;
    RwTexture *texture = RpMaterialGetTexture(material);
     
    if (texture && !RwTextureGetMaskName(texture))
    {
        DtMtlGetID(shape, group, &mtlID);    
        DtMtlGetNameByID(mtlID, &mtlName);

        texName = getTexFileNodeName(mtlName, "color");
        if(texName)
        {                
            maskName = getTexFileNodeName(mtlName, "transparency");
            if (maskName)
            {
                RwTextureSetMaskName(RpMaterialGetTexture(material), maskName);
            }
            RwFree(texName);
        }    
    }
}

MaterialMap *BuildMaterialMap(RtWorldImport *importWorld, bool verbose, bool *textureNameWarning)
{
    MaterialMap *materialMap;
    int         materialCount;
    int         i, j;

    /* Get the total number of materials in the scene */
    DtMtlGetSceneCount(&materialCount);

    if (materialCount <= 0)
    {
        return NULL;
    }

    /* Allocate a per-material map array */
    materialMap = (MaterialMap *)RwMalloc(sizeof(MaterialMap) * materialCount);
    
    if (materialMap == NULL)
    {
        return NULL;
    }

    /* Initialise the map array to do no remapping by default */
    for (i = 0; i < materialCount; i++)
    {
        materialMap[i].materialMap      = i;
        materialMap[i].rwMaterial       = NULL;
        materialMap[i].rtImportMaterial = -1;
    }

    /* Check if any materials are identical (for our purposes) */
    for (i = 0; i < materialCount; i++)
    {
        for (j = i + 1; j < materialCount; j++)
        {
            if (CompareMayaShadingGroups(i, j))
            {
                materialMap[j].materialMap = materialMap[i].materialMap;
            }
        }
    }

    /* Create RenderWare materials for those IDs that don't map to others */
    for (i = 0; i < materialCount; i++)
    {
        if (materialMap[i].materialMap == i)
        {
            materialMap[i].rwMaterial = MayaShadingGroup2RpMaterial(i, verbose, textureNameWarning); 
            
            if (importWorld != NULL)
            {
                 materialMap[i].rtImportMaterial = RtWorldImportAddMaterial(importWorld,
                                                    materialMap[i].rwMaterial);
            }
        }
    }

    return (materialMap);
}

void DestroyMaterialMap(MaterialMap *materialMap)
{
    int         materialCount;
    int         i;

    if (materialMap == NULL)
    {
        return;
    }

    /* Get the total number of materials in the scene */
    DtMtlGetSceneCount(&materialCount);

    if (materialCount <= 0)
    {
        return;
    }

    for (i = 0; i < materialCount; i++)
    {
        if (materialMap[i].rwMaterial != NULL)
        {
            RpMaterialDestroy(materialMap[i].rwMaterial);
        }
    }

    RwFree(materialMap);
}

MObject fileTextureCallBack(MObject object, void *pData)
{
    MStatus     status;
    float       k, l;
    bool        *exportKL = (bool *)pData;   

    MFnDependencyNode   depNode(object, &status);
   
    if (status != MStatus::kSuccess)
    {
        return object;
    }

    MString     objectName = depNode.name(&status);

    if (getQueryAsFloat("getAttr", (char *)objectName.asChar(), "RwK", &k, true))
    {
        *exportKL = true;
    }
    else
    {
        if (getQueryAsFloat("getAttr", (char *)objectName.asChar(), "RwL", &l, true))
        {
            *exportKL = true;
        }
    }

    return object;
}
